function [ cdata ] = agglomerate_corres( iparam, cdata ) 
% Function to agglomerate correspondences 
% using the adaptive partial linkage model of the following ICCV 2009 paper
%
% Minsu Cho, Jungmin Lee, and Kyoung Mu Lee, 
% Feature Correspondence and Deformable Object Matching via Agglomerative Correspondence Clustering, 
% Proc. International Conference on Computer Vision (ICCV), 2009.
% 
% Note that this is a simple MATLAB implementation without consideration for speed-up. 
%
% written by Minsu Cho, 2009, Seoul National University, Korea
% http://cv.snu.ac.kr/~minsucho

bPair = cdata.bPair;

ori_affinityMatrix = cdata.affinityMatrix;
clusterInfo = cdata.clusterInfo;

if iparam.bOneToOne 
    ori_affinityMatrix( find(cdata.overlapMatrix) ) = Inf;
end
affinityMatrix = ori_affinityMatrix;
nCluster = size(affinityMatrix,1);

sampleJ = 0;
while nCluster > 1 
 	sampleJ = sampleJ + 1;
    if sampleJ > iparam.nMaxIteration, break; end
 	
    % find the minimum distance
    elementVector = squareform(affinityMatrix);
    [ minValue, minIdx ] = min( elementVector );
    subs = getSubFromIdx(minIdx, size(affinityMatrix,1));
    
    % idxCluster1: the larger cluster
    if clusterInfo(subs(1)).size > clusterInfo(subs(2)).size
        idxCluster1 = subs(1);        idxCluster2 = subs(2);  
    else
        idxCluster1 = subs(2);        idxCluster2 = subs(1);
    end
    
    if minValue > iparam.cutoff,    break;    end
    
    % seed index = match index
    seedIdxInCluster1 = clusterInfo(idxCluster1).seedIdx;
    seedIdxInCluster2 = clusterInfo(idxCluster2).seedIdx;
    szCluster1 = length(seedIdxInCluster1);    szCluster2 = length(seedIdxInCluster2);
    
    if bPair
        bMergingAnA = 1;
    else
        % check whether merging A-A or A-B 
        % simply using centroid proximity.., a very naive method
        dist_AnA = sum((clusterInfo(idxCluster1).centroid_A - clusterInfo(idxCluster2).centroid_A).^2);
        dist_AnB = sum((clusterInfo(idxCluster1).centroid_A - clusterInfo(idxCluster2).centroid_B).^2);    
        bMergingAnA = dist_AnA <= dist_AnB;
    end
    
    % merge into the larger cluster
    clusterInfo(idxCluster1).seedIdx = [ seedIdxInCluster1 seedIdxInCluster2 ];
    clusterInfo(idxCluster1).size = szCluster1 + szCluster2;
    
    if ~bPair && ~bMergingAnA
        % flip matches in matchInfo
        for i=1:szCluster2
         tmpMatch = cdata.matchInfo(seedIdxInCluster2(i)).match;
         cdata.matchInfo(seedIdxInCluster2(i)).match = [ tmpMatch(2) tmpMatch(1) ];
        end
        
        % flip centroid_A & B in clusterInfo
        tmp = clusterInfo(idxCluster2).centroid_A;
        clusterInfo(idxCluster2).centroid_A = clusterInfo(idxCluster2).centroid_B;
        clusterInfo(idxCluster2).centroid_B = tmp;
    end
    % update centroids
    clusterInfo(idxCluster1).centroid_A = ( szCluster1*clusterInfo(idxCluster1).centroid_A...
        + szCluster2*clusterInfo(idxCluster2).centroid_A )/(szCluster1+szCluster2);
    clusterInfo(idxCluster1).centroid_B = ( szCluster1*clusterInfo(idxCluster1).centroid_B...
        + szCluster2*clusterInfo(idxCluster2).centroid_B )/(szCluster1+szCluster2);
    
    % eliminate the smaller cluster
    affinityMatrix(idxCluster2,:) = []; 
    affinityMatrix(:,idxCluster2) = [];        
    clusterInfo(idxCluster2) = [];
    nCluster = nCluster - 1;
    idxCluster1 = idxCluster1 - sum( idxCluster2 < idxCluster1);
    
    subAffMatrix = ori_affinityMatrix(clusterInfo(idxCluster1).seedIdx,:);
    %nCombiVec = zeros(1,nCluster); NumVec = zeros(1,nCluster);
    for cnt = 1:nCluster
        if cnt == idxCluster1,    continue;        end
        
        subSubAffMatrix = subAffMatrix(:,clusterInfo(cnt).seedIdx);
        nCombi = prod(size(subSubAffMatrix));

        % compute the adaptive partial linkage
        kNum = ceil( iparam.rNNLinkage * nCombi );
        if nCombi <= iparam.kNNLinkage
            kNum = nCombi;
        elseif kNum < iparam.kNNLinkage
            kNum = iparam.kNNLinkage;
        end
        %nCombiVec(cnt) = nCombi;   kNumVec(cnt) = kNum;
        values = subSubAffMatrix(:);
        if length(values) > kNum
            values = sort( values, 'ascend' );
        end
        affinityMatrix(idxCluster1,cnt) = mean( values(1:kNum) );
    end
    affinityMatrix(:,idxCluster1) = affinityMatrix(idxCluster1,:)';
    fprintf('> Iter#%3d: %4d clusters, prev min distance = %f, \n',sampleJ,size(affinityMatrix,1),minValue);
end

cdata.nCluster = nCluster;
cdata.clusterInfo = clusterInfo;

end

function [ sub ] = getSubFromIdx( idx, nSize )

sub = zeros( length(idx), 2);
sub(:,1) =  ceil( nSize - 0.5 - sqrt( nSize^2 - nSize + 0.25 - 2*idx ) );
sub(:,2) =  sub(:,1) + idx - (sub(:,1) - 1).*( nSize - sub(:,1)./2 );

end


